type Unit = 'KG' | 'LBS';
type ProjectLocale = "fa_IR" | "en_US" | "it_US" | "es_ES" | "pt_BR";

interface BmiResult {
  value: number;
  icon: string;
  description: string;
  title: string | ((bmi: number) => string);
  status: string;
}

const bmiRanges = {
  underweight: { max: 18 },
  normal: { min: 18, max: 25 },
  overweight: { min: 25 },
  obese: { min: 30 }
};

const localeData: Record<ProjectLanguage, Record<string, BmiResult>> = {
  en: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Focus on building muscle and a balanced diet. Stay positive and try to maintain daily caloric intake within the recommended range.",
      title: (bmi: number) => `Your BMI is ${bmi}, which is considered underweight`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "You're doing a good job maintaining your weight within the normal range. We will use your BMI to create a customized program based on your needs.",
      title: (bmi: number) => `Your BMI is ${bmi}, which is considered normal`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "You should try to focus on your weight. We will use your BMI to create a customized weight loss program.",
      title: (bmi: number) => `Your BMI is ${bmi}, which is considered overweight`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "You should try to focus on your weight. We will use your BMI to create a customized weight loss program.",
      title: (bmi: number) => `Your BMI is ${bmi}, which is considered obese`,
      status: "overweight",
    },
  },
  it: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentrati su tonificare i muscoli e su una dieta equilibrata. Sii positivo e cerca di mantenere il consumo calorico quotidiano nell'intervallo raccomandato.",
      title: (bmi: number) => `Il tuo IMC è ${bmi}, che è considerato sottopeso`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "Ottimo lavoro! Continua a mantenere il tuo stile di vita sano con esercizio regolare e una dieta equilibrata.",
      title: (bmi: number) => `Il tuo IMC è ${bmi}, che è considerato peso normale`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentrati su tonificare i muscoli e su una dieta equilibrata. Cerca di ridurre l'apporto calorico per evitare l'aumento di peso.",
      title: (bmi: number) => `Il tuo IMC è ${bmi}, che è considerato sovrappeso`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentrati su tonificare i muscoli e su una dieta equilibrata. Cerca di ridurre l'apporto calorico per evitare l'aumento di peso.",
      title: (bmi: number) => `Il tuo IMC è ${bmi}, che è considerato sovrappeso`,
      status: "overweight",
    },
  },
  fa: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "برای ساخت عضله و رژیم غذایی متعادل تمرکز کنید. مثبت بمانید و سعی کنید مصرف کالری روزانه را در محدوده توصیه شده نگه دارید.",
      title: (bmi: number) => `شاخص توده بدنی شما ${bmi} است که به عنوان کمبود وزن در نظر گرفته می‌شود.`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "عالی است! به حفظ سبک زندگی سالم خود ادامه دهید، با ورزش منظم و یک رژیم غذایی متعادل.",
      title: (bmi: number) => `شاخص توده بدنی شما ${bmi} است که به عنوان وزن نرمال در نظر گرفته می‌شود.`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "تمرکز خود را روی تقویت عضلات و حفظ یک رژیم غذایی سالم بگذارید. مصرف کالری خود را برای جلوگیری از اضافه وزن زیاد تنظیم کنید.",
      title: (bmi: number) => `شاخص توده بدنی شما ${bmi} است که به عنوان اضافه وزن در نظر گرفته می‌شود.`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "تمرکز خود را روی تقویت عضلات و حفظ یک رژیم غذایی سالم بگذارید. مصرف کالری خود را برای جلوگیری از اضافه وزن زیاد تنظیم کنید.",
      title: (bmi: number) => `شاخص توده بدنی شما ${bmi} است که به عنوان اضافه وزن در نظر گرفته می‌شود.`,
      status: "overweight",
    },
  },
  Sp: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concéntrese en construir músculo y mantener una dieta equilibrada. Manténgase positivo y trate de mantener su ingesta calórica diaria dentro del rango recomendado.",
      title: (bmi: number) => `Tu IMC es ${bmi}, lo cual se considera bajo peso`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "¡Buen trabajo! Continúa manteniendo tu estilo de vida saludable con ejercicio regular y una dieta equilibrada.",
      title: (bmi: number) => `Tu IMC es ${bmi}, lo cual se considera peso normal`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concéntrese en tonificar los músculos y mantener una dieta equilibrada. Considere ajustar su ingesta calórica para evitar un aumento excesivo de peso.",
      title: (bmi: number) => `Tu IMC es ${bmi}, lo cual se considera sobrepeso`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concéntrese en tonificar los músculos y mantener una dieta equilibrada. Considere ajustar su ingesta calórica para evitar un aumento excesivo de peso.",
      title: (bmi: number) => `Tu IMC es ${bmi}, lo cual se considera sobrepeso`,
      status: "overweight",
    },
  },
  Br: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "É hora de priorizar o ganho de peso saudável. O Matcha AI usará essas informações para criar um programa personalizado e direcionado de ganho de peso adaptado às suas necessidades.",
      title: (bmi: number) => `Seu IMC é ${bmi}, indicando que você está abaixo do peso.`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "Você já está em um bom nível de condicionamento físico—agora é o momento de avançar para o próximo nível e trabalhar de forma mais profissional para alcançar o corpo ideal.",
      title: (bmi: number) => `Seu IMC é ${bmi}, colocando você na faixa normal.`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Para ajudá-lo/a a alcançar um estilo de vida mais saudável, o Matcha AI usará essas informações para criar um plano personalizado de perda de peso, baseado em ciência e adaptado às suas necessidades específicas.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que indica que você está na faixa de sobrepeso.`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentre-se em tonificar os músculos e manter uma dieta equilibrada. Considere ajustar sua ingestão calórica para evitar ganho de peso excessivo.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que é considerado sobrepeso`,
      status: "overweight",
    },
  },
  PL: {
    underweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentre-se em construir músculos e manter uma dieta equilibrada. Mantenha-se positivo e tente manter sua ingestão calórica diária dentro da faixa recomendada.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que é considerado abaixo do peso`,
      status: "underweight",
    },
    normal: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/ok-hand.png",
      description: "Ótimo trabalho! Continue mantendo seu estilo de vida saudável com exercícios regulares e uma dieta equilibrada.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que é considerado peso normal`,
      status: "normal",
    },
    overweight: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentre-se em tonificar os músculos e manter uma dieta equilibrada. Considere ajustar sua ingestão calórica para evitar ganho de peso excessivo.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que é considerado sobrepeso`,
      status: "overweight",
    },
    obese: {
      value: 0,
      icon: "https://api.matcha-ai.fit/files/landingpics/warn.png",
      description: "Concentre-se em tonificar os músculos e manter uma dieta equilibrada. Considere ajustar sua ingestão calórica para evitar ganho de peso excessivo.",
      title: (bmi: number) => `Seu IMC é ${bmi}, o que é considerado sobrepeso`,
      status: "overweight",
    },
  },
};

export function getBmi(locale: ProjectLanguage, weight: number, height: number, weightUnit: Unit): BmiResult {
  const heightInMeters = height / 100;

  let weightInKg: number;
  if (weightUnit === 'KG') {
    weightInKg = weight;
  } else if (weightUnit === 'LBS') {
    weightInKg = weight * 0.453592;
  } else {
    throw new Error('Invalid weight unit');
  }

  const bmi = weightInKg / (heightInMeters * heightInMeters);
  const roundedBmi = Math.round(bmi);

  let status = 'normal';
  if (roundedBmi <= bmiRanges.underweight.max) {
    status = 'underweight';
  } else if (roundedBmi > bmiRanges.normal.max) {
    status = 'overweight';
  }

  const result = localeData[locale][status];
  const title = typeof result.title === 'function' ? result.title(roundedBmi) : result.title;

  return {
    ...result,
    value: roundedBmi,
    title,
  };
}
