import { QUESTION } from "@/types/questions";
import { ApiManager } from "@/utils/axios.config";
import {
  Decode,
  DecodePertionDate,
  parseQueryParams,
} from "@/utils";
import { redirect } from "next/navigation";
import { cookies } from "next/headers";

type question = {
  data: QUESTION;
  message: string;
  result: QUESTION;
  success: string;
};

type getQuestionProps = {
  params: string;
  Authorization: string | undefined;
  userCode: string | undefined;
  type: string | undefined;
  ip: string;
  referral: string | undefined;
  phone: string | undefined;
  platform: "server" | "iOS" | "android" | "webApp"
};


function safeDecodeBase64(input?: string): string | undefined {
  try {
    return input ? atob(input) : undefined;
  } catch {
    return undefined;
  }
}

function safeDecodeURIComponent(input?: string): string | undefined {
  try {
    return input ? decodeURIComponent(input) : undefined;
  } catch {
    return undefined;
  }
}

function extractLanguageCode(typeParam: string) {
  const parts = typeParam.split("_");
  return parts.length > 1 ? (parts[1] as ProjectLanguage) : "fa";
}

export const getQuestion = async ({
  params,
  Authorization,
  type,
  userCode,
  ip,
  platform,
  referral,
  phone
}: getQuestionProps) => {
  try {
    let queryParams: any = {};
    try {
      queryParams = parseQueryParams(Decode(params));
    } catch (e) {
      redirect("/error");
    }
    const data: Record<string, string | undefined> = {
      type: queryParams.type ?? type,
      Authorization,
      code: userCode,
      aid: queryParams?.aid,
      height_cm: safeDecodeBase64(queryParams?.height_cm),
      height_ft: safeDecodeBase64(queryParams?.height_ft),
      height_in: safeDecodeBase64(queryParams?.height_in),
      weight_current_kg: safeDecodeBase64(queryParams?.weight_current_kg),
      weight_current_lbs: safeDecodeBase64(queryParams?.weight_current_lbs),
      weight_goal_lbs: safeDecodeBase64(queryParams?.weight_goal_lbs),
      weight_goal_kg: safeDecodeBase64(queryParams?.weight_goal_kg),
      carbs: safeDecodeBase64(queryParams?.carbs),
      fat: safeDecodeBase64(queryParams?.fat),
      calory: safeDecodeBase64(queryParams?.calory),
      protein: safeDecodeBase64(queryParams?.protein),
      redirect: queryParams?.redirect,
      calendar: DecodePertionDate(queryParams?.calendar),
      email: safeDecodeURIComponent(queryParams.email),
      age: safeDecodeBase64(queryParams.age),
      v_variable: safeDecodeBase64(queryParams.v_variable),
      referral,
      phone
    };

    const finalParams = Object.fromEntries(
      Object.entries(data).filter(([_, v]) => v !== undefined && v !== "" && v !== "undefined")
    );

    const language = extractLanguageCode(finalParams.type ?? "");

    const finalQueryParams = {
      ...finalParams,
      version: "v3",
      language,
      ip,
      platform,
    };

    const response = await ApiManager.get<question>("/main/question.php", {
      params: finalQueryParams,
    });
    console.log({ finalQueryParams: finalQueryParams })
    if (response.data?.success === "true") {
      const data = response.data.data
      // console.log({ data })
      if (Array.isArray(data) || !data || typeof (data) !== "object")
        throw new Error("Unexpected API format", { cause: data })
      return response.data.data;
    }

    if (response.data?.success === "false") {
      console.warn("API responded with failure:", response.data.message);
      redirect("/error");
    }

    // Fallback: unexpected but no error thrown
    console.warn("Unexpected API format:", response.data);
    redirect("/error");

  } catch (error) {
    console.error("Unhandled error in getQuestion:", error);
    redirect("/error");
  }
};