'use client'; 

import React, { useState, useEffect } from 'react';
import styled from 'styled-components';
import 'swiper/css';
import 'swiper/css/pagination';
import Image from 'next/image';
import type { Swiper as SwiperType } from 'swiper';
import { Pagination } from 'swiper/modules';
import { Swiper, SwiperSlide } from 'swiper/react';
import { AppFlex, Button, Typography } from '@/primitives';
import { callFlutterHandlerOrFallback } from '@/utils/callFlutterHandlerOrFallback';
import { usePlatform } from '@/hooks';
import { THISPROJECT } from "@/constants/projects";

interface SliderItem {
  key: string;
  image: string;
  title: string; 
  description: string;
}

interface SliderData {
  items: SliderItem[];
}

type Props = {
  handleClick: () => void;
  inputs: SliderData; 
  buttonText: string;
};

export default function SliderRoot({ handleClick, inputs, buttonText }: Props) {
  const [activeIndex, setActiveIndex] = useState(0);
  const { isNotWeb, isWeb, platform } = usePlatform()

    useEffect(() => {
    if (isNotWeb) {
      callFlutterHandlerOrFallback({
        flutterHandlerName: "sendCustomParams",
        flutterHandlerParams: ["tiktok", THISPROJECT.TIKTOK_PIXEL_ID],
      });
    }
  }, [isNotWeb]);

  // Update activeIndex when slide changes
  const handleSlideChange = (swiper: SwiperType) => setActiveIndex(swiper.activeIndex);

  return (
    <SliderContainer>
      <SlideImage
        src={inputs.items[activeIndex]?.image}
        alt={inputs.items[activeIndex]?.title}
        width={598}
        height={600}
        style={{
          objectFit: 'cover',
          position: "fixed",
          maxWidth: "100%",
          top: 0,
          left: "50%",
          transform: "translateX(-50%)"
        }}
        priority={true}
      />
      {/* Bottom Section - Contains Swiper for title and description */}
      <BottomSection>
        <StyledSwiper
          modules={[Pagination]}
          pagination={{ clickable: true, horizontalClass: 'slider-parasite-bullets' }}
          spaceBetween={0}
          slidesPerView={1}
          onSlideChange={handleSlideChange}
          allowTouchMove={true}
        >
          {inputs.items?.map((slide) => (
            <SwiperSlide key={slide.key}>
              <ContentWrapper>
                <Title>{slide.title}</Title>
                <Description>{slide.description}</Description>
              </ContentWrapper>
            </SwiperSlide>
          ))}
        </StyledSwiper>

        {/* Action Button */}
        <Button variant="question" onClick={handleClick} position="fixed" key="button">
          <Typography>{buttonText}</Typography>
        </Button>
      </BottomSection>
    </SliderContainer>
  );
}

const SliderContainer = styled.div`
  position: relative;
  height: calc(100vh - 200px);
  overflow: hidden;
`;

const BottomSection = styled.div`
  position: fixed;
  bottom: 0;
  left: 50%;
  transform: translateX(-50%);
  width: 100%;
  max-width: 600px;
  height: 45vh;
  background: white;
  display: flex;
  flex-direction: column;
  gap: 16px;
  justify-content: center; /* Centers content vertically */
  align-items: center;
  padding: 40px 20px 20px 20px;
  border-radius: 40px 40px 0 0;
  box-shadow: 0 -4px 10px rgba(0, 0, 0, 0.1);
  border: 1px solid #ccc;
  z-index: 2;
  pointer-events: auto; /* Ensures BottomSection captures pointer events */
`;

const StyledSwiper = styled(Swiper)`
  width: 100% ;
  height: 100%;

  .swiper-pagination {
    /* Customize pagination bullets if needed */
  }

  .swiper-wrapper {
    display: flex;
  }

  .swiper-slide {
    display: flex;
    justify-content: center;
    align-items: center;
    /* Prevents slide content from shifting */
  }
`;

const ContentWrapper = styled(AppFlex)`
  flex-direction: column;
  height: 100%;
  gap: 16px;
  text-align: center;
`;

const SlideImage = styled(Image)`
  object-fit: cover;
`;

const Title = styled.h2`
  font-size: 24px;

  @media (max-width: 768px) {
    font-size: 20px;
  }
`;

const Description = styled.p`
  font-size: 16px;
  color: #555;

  @media (max-width: 768px) {
    font-size: 14px;
  }
`;
