"use client";

import { Button, RangeInput, Typography } from "@/primitives";
import { useQuestionContext } from "@/providers";
import { colors } from "@/theme";
import { RangeSliderElementInputs } from "@/types/questions";
import { Encode } from "@/utils";
import Image from "next/image";
import { useCallback, useEffect, useMemo, useState, type FC } from "react";
import styled from "styled-components";

interface RangeSliderProps {
  inputs: RangeSliderElementInputs;
  buttonText: string;
  aid: string;
}

const RangeSlider: FC<RangeSliderProps> = ({ inputs: { items }, buttonText, aid }) => {
  const { dispatch } = useQuestionContext();
  const [rangeValue, setRangeValue] = useState("0");

  const onChangeRange = (value: string) => {
    setRangeValue(value);
  };

  // get label value
  const getLabelValue = useMemo(() => {
    const value = +rangeValue;
    const steps = items.steps;
    const stepSize = 100 / (steps.length - 1);

    const index = value / stepSize;
    const lowerIndex = Math.floor(index);
    const upperIndex = Math.ceil(index);

    if (lowerIndex === upperIndex) {
      return Number(steps[lowerIndex].value);
    }

    const lowerValue = Number(steps[lowerIndex].value);
    const upperValue = Number(steps[upperIndex].value);
    const fraction = index - lowerIndex;

    return Math.round((lowerValue + (upperValue - lowerValue) * fraction) * 10) / 10;
  }, [rangeValue, items.steps]);

  // get active icon
  const isActiveIcon = useCallback(
    (iconIndex: number) => {
      const value = +rangeValue;
      const steps = items.steps;
      const stepSize = 100 / (steps.length - 1);
      const currentStep = Math.round(value / stepSize);
      return currentStep === iconIndex;
    },
    [rangeValue, items.steps]
  );

  // get step caption
  const getStepCaption = useMemo(() => {
    const value = +rangeValue;
    const steps = items.steps;
    const stepSize = 100 / (steps.length - 1);

    if (value === 0) {
      return steps[0].caption;
    }

    for (let i = 1; i < steps.length; i++) {
      if (value <= stepSize * i) {
        return steps[i].caption;
      }
    }

    return steps[steps.length - 1].caption;
  }, [rangeValue, items.steps]);


  // handle submit
  const handleSubmit = () => {
    dispatch({
      type: "UPDATE_QUESTIONS",
      payload: {
        aid,
        v_variable: Encode(getLabelValue.toString()),
      },
    });
  };

  // set range value to the first step
  useEffect(() => {
    const stepSize = 100 / (items.steps.length - 1);
    setRangeValue(stepSize.toString());
  }, []);

  return (
    <Root>
      <Title>{items.title}</Title>
      <RangeValue>
        {getLabelValue} {items.unit}
      </RangeValue>

      {/* top tickers */}
      <Tickers>
        {items.steps.map((ticker, index) => (
          <StyledImage
            $isActive={isActiveIcon(index)}
            key={`top-ticker-${index}`}
            src={ticker.icon}
            width={70}
            height={60}
            alt=""
          />
        ))}
      </Tickers>

      {/* range input */}
      <RangeInput
        onRangeChange={onChangeRange}
        value={rangeValue}
        step={items.step || 100 / (items.steps.length - 1)}
      />

      {/* bottom tickers */}
      <Tickers>
        {items.steps.map((ticker, index) => (
          <TickerTitle className="ticker-title" key={`bottom-ticker-${index}`}>
            {ticker.value} {items.unit}
          </TickerTitle>
        ))}
      </Tickers>

      {/* step caption */}
      <StepCaption>{getStepCaption}</StepCaption>

      {/* action button */}
      <Button variant="question" onClick={handleSubmit} position={"fixed"} key={"button"}>
        <Typography>{buttonText}</Typography>
      </Button>
    </Root>
  );
};

export default RangeSlider;

const Root = styled.div`
  display: flex;
  flex-direction: column;
  gap: 30px;
  direction: ltr;
`;

const Title = styled.p`
  font-size: 16px;
  text-align: center;
  margin-top: 20px;
`;

const StyledImage = styled(Image) <{ $isActive: boolean }>`
  filter: ${({ $isActive }) => ($isActive ? "none" : "grayscale(100%) brightness(30%)")};
  transition: filter 0.3s ease-in-out;
`;

const Tickers = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
`;

const TickerTitle = styled.div`
  font-size: 16px;
  font-weight: 600;
`;

const RangeValue = styled.div`
  text-align: center;
  font-size: 28px;
  font-weight: bold;
`;

const StepCaption = styled.div`
  margin-top: 30px;
  text-align: center;
  font-size: 16px;
  font-weight: 600;
  background-color: #d2d0d0;
  padding: 20px 10px;
  border-radius: 15px;
  color: ${colors.black};
`;
