import React, { useState, useEffect } from 'react';
import styled from 'styled-components';
import { AppFlex, Typography } from '@/primitives';
import Image from 'next/image';
import { CircularProgressbar, buildStyles } from 'react-circular-progressbar';
import { ProSheet, useModalController } from 'react-modal-pro';
import NutritionInfoText from './Text';
import LinearProgressBar from './LinearProgressbar';

interface NutritionProgressCardProps {
  id?: string;
  title: string;
  value: string;
  maxValue?: string;
  unit?: string;
  progressColor: string;
  icon?: string;
  isHealthScore?: boolean;
  editTitle?: string;
  onNutritionChange?: (field: keyof {
    carbs: string;
    calory: string;
    protein: string;
    fat: string;
  }, value: string) => void;
}

const NutritionProgressCard: React.FC<NutritionProgressCardProps> = (props) => {
  const { id, title, value, maxValue, unit, progressColor, icon, isHealthScore = false, editTitle = "", onNutritionChange } = props;
  const { handleCloseModal } = useModalController(`edit-nutrition-${id}`)
  const percentage = isHealthScore
    ? Math.min(Number(value) * 10, 100)
    : Math.min((Number(value) / Number(maxValue || 1)) * 100, 100);

  const [inputValue, setInputValue] = useState(value);

  useEffect(() => {
    setInputValue(value);
  }, [value]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setInputValue(e.target.value);
  };

  const handleDone = () => {
    if (onNutritionChange && id) {
      onNutritionChange(id as keyof {
        carbs: string;
        calory: string;
        protein: string;
        fat: string;
      }, inputValue);
    }
    handleCloseModal()
  };

  const handleRevert = () => {
    setInputValue(value);
    handleCloseModal
  };

  return (
    <StyledNutritionProgressCard fullWidth={isHealthScore}>
      {isHealthScore ? (
        <HealthscoreContainer>
          {icon && (
            <Image
              src={icon}
              alt={title}
              width={24}
              height={24}
            />
          )}
          <AppFlex style={{ width: '100%', flexDirection: "column", gap: "4px", alignItems: "flex-start" }}>
            <NutritionInfoText
              text={title}
              textAlign="center"
              textSize="14px"
              textColor="#1E1E1E"
              textWeight="500"
            />
            <LinearProgressBar
              value={Number(value)}
              max={10}
              trackColor="#E4E4EE"
              fillColor="#37CAA5"
              height="4px"
              borderRadius="8px"
            />
          </AppFlex>
        </HealthscoreContainer>
      ) : (
        <>
          <ProSheet
            direction="bottom"
            modalKey={`edit-nutrition-${id}`}
            TriggerElement={
              <Image
                alt="edit"
                src={'/svg/edit-2.svg'}
                width={16}
                height={16}
                style={{ position: 'absolute', bottom: '10px', right: '10px' }}
              />
            }
          >
            <AppFlex direction="column" gap="24px">
              <Typography style={{ fontSize: '24px', fontWeight: '600' }}>
                {editTitle}
              </Typography>
              <AppFlex
                style={{
                  width: '100%',
                  borderRadius: '16px',
                  border: '1px solid #F4F5F7',
                  padding: '16px',
                }}
              >
                <ProgressCircleContainer>
                  <CircularProgressbar
                    value={percentage}
                    text={`${inputValue}${unit}`}
                    strokeWidth={4}
                    styles={buildStyles({
                      textSize: '16px',
                      pathColor: progressColor || '#4CAF50',
                      textColor: '#000',
                      trailColor: '#d6d6d6',
                      backgroundColor: '#fff',
                    })}
                    aria-label={`${title}: ${inputValue}${unit}`}
                  />
                </ProgressCircleContainer>
              </AppFlex>
              <label htmlFor="nutrition" style={{ width: '100%' }}>
                <Typography
                  style={{ padding: '4px', fontSize: '12px', color: '#48454E' }}>
                  {title}
                </Typography>
                <input
                  name="nutrition"
                  type="text"
                  value={inputValue}
                  onChange={handleInputChange}
                  style={{
                    color: '#48454E',
                    padding: '16px',
                    borderRadius: '16px',
                    width: '100%',
                  }}
                />
              </label>
              <AppFlex
                align="center"
                justify="center"
                gap="8px"
                style={{ width: '100%' }}
              >
                <button
                  style={{
                    width: '100%',
                    padding: '16px 51px',
                    background: '#fff',
                    border: '2px solid #37CAA5',
                    borderRadius: '32px',
                    fontSize: '18px',
                    fontWeight: '500',
                    color: '#1E1E1E',
                  }}
                  onClick={() => handleRevert()}
                >
                  Revert
                </button>
                <button
                  style={{
                    width: '100%',
                    padding: '16px 51px',
                    background: '#37CAA5',
                    border: '2px solid #37CAA5',
                    borderRadius: '32px',
                    fontSize: '18px',
                    fontWeight: '500',
                    color: '#FFFFFF',
                  }}
                  onClick={() => handleDone()}
                >
                  Done
                </button>
              </AppFlex>
            </AppFlex>
          </ProSheet>
          <ProgressCircleContainer>
            <div style={{ minHeight: '16px' }}>
              <NutritionInfoText
                text={title}
                textAlign="center"
                textSize="14px"
                textColor="#555"
                textWeight="500"
                style={{
                  marginTop: '8px',
                  position: "absolute",
                  top: "2px",
                  right: "10px"
                }}
              />
            </div>
            <CircularProgressbar
              value={percentage}
              text={`${value}${unit}`}
              strokeWidth={4}
              styles={buildStyles({
                textSize: '16px',
                pathColor: progressColor || '#4CAF50',
                textColor: '#000',
                trailColor: '#d6d6d6',
                backgroundColor: '#fff',
              })}
              aria-label={`${title}: ${value}${unit}`}
            />
          </ProgressCircleContainer>

        </>
      )}
    </StyledNutritionProgressCard>
  );
};

export default NutritionProgressCard;

const StyledNutritionProgressCard = styled(AppFlex) <{ fullWidth?: boolean }>`
  background: #ffffff;
  position: relative;
  padding: 16px;
  border-radius: 12px;
  box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
  flex-direction: column;
  align-items: center;
  text-align: center;
  max-width: 100%;

  ${({ fullWidth }) =>
    fullWidth &&
    `
    grid-column: span 2;
    display: flex;
    justify-content: center;
    align-items: center;
  `}
`;

const ProgressCircleContainer = styled.div`
  width: 72%;
  margin-left: 0;
  margin-right: auto;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;

  .CircularProgressbar {
    width: 100%;
    height: 100%;
  }

  .CircularProgressbar-path {
    stroke-width: 4px !important;
  }

  .CircularProgressbar-trail {
    stroke-width: 4px !important;
  }

  .CircularProgressbar-text {
    font-size: 16px !important;
    dominant-baseline: middle;
    text-anchor: middle;
  }
`;

const HealthscoreContainer = styled(AppFlex)`
  width: 100%;
  align-items: start;
  justify-content: center;
  gap: 8px;
`;
