import React from 'react';
import styled from 'styled-components';

interface LinearProgressBarProps {
  value: number;
  max: number;
  text?: string;
  trackColor?: string;
  fillColor?: string;
  height?: string;
  borderRadius?: string;
}


const LinearProgressBar: React.FC<LinearProgressBarProps> = ({
  value,
  max,
  text,
  trackColor = '#d6d6d6',
  fillColor = '#4CAF50',
  height = '10px',
  borderRadius = '5px',
}) => {
  const percentage = Math.min((value / max) * 100, 100);

  return (
    <ProgressContainer>
      <StyledRange
        type="range"
        min="0"
        max={max}
        value={percentage}
        fillColor={fillColor}
        trackColor={trackColor}
        height={height}
        borderRadius={borderRadius}
        valuePercentage={String(percentage)}
      />
      {text && <ProgressText>{text}</ProgressText>}
    </ProgressContainer>
  );
};

export default LinearProgressBar;


const StyledRange = styled.input<{
  fillColor: string;
  trackColor: string;
  height: string;
  borderRadius: string;
  valuePercentage: string
}>`
  -webkit-appearance: none;
  width: 100%;
  height: ${({ height }) => height};
  border-radius: ${({ borderRadius }) => borderRadius};
  background: ${({ trackColor }) => trackColor};
  outline: none;
  pointer-events: none; /* Make it non-interactive */

  &::-webkit-slider-thumb {
    -webkit-appearance: none;
    appearance: none;
    width: 0; /* Hide the thumb */
    height: 0;
    background: transparent;
    cursor: pointer;
  }

  &::-moz-range-thumb {
    width: 0;
    height: 0;
    background: transparent;
    cursor: pointer;
  }

  /* Filled part */
  background: linear-gradient(
    to right,
    ${({ fillColor }) => fillColor} 0%,
    ${({ fillColor }) => fillColor} ${({ valuePercentage: number }) => `${number}%`},
    ${({ trackColor }) => trackColor} ${({ valuePercentage: number }) => `${number}%`},
    ${({ trackColor }) => trackColor} 100%
  );
`;

const ProgressContainer = styled.div`
  width: 100%;
  position: relative;
  display: flex;
  align-items: center;
`;

const ProgressText = styled.span`
  position: absolute;
  width: 100%;
  text-align: center;
  font-size: 14px;
  color: #555;
`;