"use client";

import { AppFlex, Typography } from "@/primitives";
import React, { useRef, useState, useEffect } from "react";

type VideoPlayerProps = {
  videoUrl: string;
  videoTitle?: string
};

const PreCheckoutVideoPlayer: React.FC<VideoPlayerProps> = ({ videoUrl, videoTitle = "" }) => {
  const videoRef = useRef<HTMLVideoElement>(null);
  const [isPlaying, setIsPlaying] = useState(false);
  const [progress, setProgress] = useState(0);
  const [duration, setDuration] = useState(0);

  // Play/Pause Handler
  const handlePlayPause = () => {
    if (!videoRef.current) return;
    if (videoRef.current.paused) {
      videoRef.current.play();
      setIsPlaying(true);
    } else {
      videoRef.current.pause();
      setIsPlaying(false);
    }
  };

  // Update Progress Bar as Video Plays
  const handleTimeUpdate = () => {
    if (videoRef.current) {
      setProgress(videoRef.current.currentTime);
    }
  };

  // Set Video Duration on Load
  const handleLoadedMetadata = () => {
    if (videoRef.current) {
      setDuration(videoRef.current.duration);
    }
  };

  // Seek Video when User Moves Slider
  const handleSeek = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (videoRef.current) {
      const newTime = parseFloat(e.target.value);
      videoRef.current.currentTime = newTime;
      setProgress(newTime);
    }
  };

  // Reset Play Button when Video Ends
  useEffect(() => {
    const videoElement = videoRef.current;
    const handleEnded = () => setIsPlaying(false);
    if (videoElement) {
      videoElement.addEventListener("ended", handleEnded);
    }
    return () => {
      if (videoElement) {
        videoElement.removeEventListener("ended", handleEnded);
      }
    };
  }, []);

  return (
    <AppFlex direction="column" gap="5px" align="center">
      <Typography color="#767A86" fontSize={16} fontWeight={600} lineHeight={1.5} style={{ marginTop: "10px" }}>
        {videoTitle}
      </Typography>
      <div
        style={{
          position: "relative",
          width: "100%",
          borderRadius: "10px",
          overflow: "hidden",
          borderRight: '8px'
        }}
      >
        {/* Video Element (No Default Controls) */}
        <video
          ref={videoRef}
          src={videoUrl}
          style={{ width: "100%" }}
          onClick={handlePlayPause}
          onTimeUpdate={handleTimeUpdate}
          onLoadedMetadata={handleLoadedMetadata}
        />

        {/* Custom Play Button */}
        {!isPlaying && (
          <div
            onClick={handlePlayPause}
            style={{
              position: "absolute",
              top: "50%",
              left: "50%",
              transform: "translate(-50%, -50%)",
              cursor: "pointer",
              zIndex: 2,
              width: "80px",
              height: "80px",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
              background: "rgba(0, 0, 0, 0.5)",
              borderRadius: "50%",
            }}
          >
            <svg width="40" height="40" viewBox="0 0 24 24" fill="white">
              <path d="M8 5v14l11-7z" />
            </svg>
          </div>
        )}

        {/* Custom Seekbar */}
        <div
          style={{
            position: "absolute",
            bottom: "10px",
            left: "50%",
            transform: "translateX(-50%)",
            width: "90%",
            zIndex: 3,
            display: "flex",
            alignItems: "center",
            direction: "ltr",
          }}
        >
          <input
            type="range"
            min={0}
            max={duration}
            step="0.1"
            value={progress}
            onChange={handleSeek}
            style={{
              width: "100%",
              appearance: "none",
              height: "5px",
              background: `linear-gradient(to right, #3AD7AB ${(progress / duration) * 100
                }%, #ddd ${(progress / duration) * 100}%)`, // Red progress
              borderRadius: "5px",
              outline: "none",
              transition: "background 0.1s linear",
            }}
          />
        </div>
      </div>
    </AppFlex>
  );
};

export default PreCheckoutVideoPlayer;
