import type { Metadata, Viewport } from "next";
import {
  GoogleAnalytics,
  LanguageProvider,
  ReactQueryProvider,
  StyledComponentsRegistry,
} from "@/providers";
import "../src/theme/globals.css";
import { GlobalStyle } from "@/theme";
import RootLayout from "@/layouts/RootLayout";
import ModalProvider from "@/providers/ModalProvider";
import { en_USFonts, fa_IRFonts } from "@/theme/fonts";
import { THISPROJECT, thisLocale } from "@/constants/projects";
import { getLanguageFromCookiesServer } from "@/utils/question/getLanguage.server";

declare global {
  interface Window {
    gtag: any;
    getDevs: () => void;
    paypal: any;
    flutter_inappwebview?: {
      callHandler: (handlerName: string, ...args: any[]) => Promise<void>;
    };
  }
}

export const metadata: Metadata = {
  title: THISPROJECT.PAGE_TITLE,
  description: THISPROJECT.PAGE_DESCRIPTION,
  icons: THISPROJECT.FAVICON_URL,
};

export const viewport: Viewport = {
  width: "device-width",
  initialScale: 1,
  maximumScale: 1,
  userScalable: false,
};

export default async function MainLayout({ children }: { children: React.ReactNode }) {
  const direction = THISPROJECT.DEFAULT_LOCALE === "fa_IR" ? "rtl" : "ltr";
  const locale = thisLocale;
  const fonts = thisLocale === "en_US" || thisLocale === "it_US" ? en_USFonts : fa_IRFonts;
  const language: ProjectLanguage = await getLanguageFromCookiesServer();
  return (
    <html dir={direction}>
      <head>
        <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
        {/* <TikTokPixel /> */}
      </head>
      <body
        translate="no"
        style={{
          fontFamily: `${fonts.style.fontFamily}`,
        }}
      >
        <GoogleAnalytics />
        <ReactQueryProvider>
          <StyledComponentsRegistry>
            <GlobalStyle locale={locale} />
            <RootLayout>
              <LanguageProvider initialLanguage={language}>
                <ModalProvider>
                  {children}
                </ModalProvider>
              </LanguageProvider>
            </RootLayout>
          </StyledComponentsRegistry>
        </ReactQueryProvider>
      </body>
    </html>
  );
}
