"use client";

import { sendPhoneNumber, sendVerificationCode } from "@/components/confirm-pay/confirmPay.services";
import { THISPROJECT } from "@/constants/projects";
import { useCountDownTimer } from "@/hooks";
import Header from "@/layouts/header/Header";
import { AppFlex, Button, Typography } from "@/primitives";
import { imageLoader } from "@/utils";
import { useMutation } from "@tanstack/react-query";
import Image from "next/image";
import { useSearchParams } from "next/navigation";
import React, { useEffect, useState } from "react";
import { useForm } from "react-hook-form";
import { IoMdInformationCircleOutline, IoMdLock } from "react-icons/io";
import { MdOutlineModeEdit } from "react-icons/md";
import ReactLoading from "react-loading";
import { toast } from "react-toastify";
import { styled } from "styled-components";
import Cookies from "universal-cookie";

interface OTPCredential extends Credential {
  code: string;
}

interface OTPCredentialRequestOptions {
  transport: string[];
}

interface ExtendedCredentialRequestOptions extends CredentialRequestOptions {
  otp?: OTPCredentialRequestOptions;
}

declare global {
  interface Window {
    OTPCredential?: any;
  }
}

type Props = {};

const ConfirmPAyPage = (props: Props) => {
  /**
   * const and variables
   * _______________________________________________________________________________
   */
  const time = "120000";
  const cookie = new Cookies();
  const token = cookie.get("Authorization");
  const searchParams = useSearchParams();
  const code = searchParams.get("code");
  const [step, setStep] = useState<"verificationcode" | "recivecode">("recivecode");
  const [userData, setUserData] = useState({
    phNumber: "",
  });
  const [, { start, render, isStarted, reset: resetCount, pause }] = useCountDownTimer(Number(time), 1000, "minutes");

  const {
    register,
    reset,
    watch,
    setValue,
    formState: { errors },
  } = useForm({
    defaultValues: {
      mobileNumber: "",
      VerificationCode: "",
    },
    mode: "onChange",
  });

  /**
   * hooks and methods
   * _______________________________________________________________________________
   */

  const handleWebOTP = () => {
    if (typeof window !== 'undefined' && 'OTPCredential' in window && navigator.credentials) {
      const ac = new AbortController();

      const options: ExtendedCredentialRequestOptions = {
        otp: { transport: ['sms'] },
        signal: ac.signal
      };

      navigator.credentials.get(options)
        .then((otp) => {
          const otpCredential = otp as OTPCredential;
          if (otpCredential && otpCredential.code) {
            const cleanCode = otpCredential.code;
            setValue("VerificationCode", cleanCode);
          }
        })
        .catch(err => {
          console.log('WebOTP error:', err);
        });

      setTimeout(() => ac.abort(), 120000);
    }
  };

  const { mutate: sendCodeMutate, isLoading: loadingMutate } = useMutation(
    (variables: { mobileNumber: string }) => {
      return sendPhoneNumber(Number(step === "recivecode" ? watch("mobileNumber") : userData.phNumber), window.location.href, token);
    },
    {
      onSuccess: data => {
        if (data.success === "true") {
          setStep("verificationcode");
          setUserData({ phNumber: data.data.phoneNumber });
          cookie.set("phoneNumber", data.data.phoneNumber, { path: "/" });
          toast.success(data.data.verifySubtitle);
          if (step === "verificationcode") {
            start();
          }
        }
        if (data.success === "false") {
          toast.error(data.message);
        }
      },
      onError: error => console.log(error, "error"),
    }
  );

  const { mutate, isLoading } = useMutation(
    (variables: { mobileNumber: string; verificationCode: string; code: string | null }) => {
      return sendVerificationCode(userData.phNumber, watch("VerificationCode") as string, code as string, window.location.href);
    },
    {
      onSuccess: data => {
        if (data.success === "true") {
          toast.success(data.data.verifySubtitle);
          window.location.assign(data?.data?.resultLink);
          cookie.set("phoneNumberVerified", true, { path: "/" });
        }
        if (data.success === "true" && data.message === "404") {
          toast.success(data.message);
        }
        if (data.success === "false") {
          toast.error(data.message);
        }
      },
      onError: error => {
        console.log(error);
      },
    }
  );

  const handleResetCode = () => {
    sendCodeMutate({ mobileNumber: userData.phNumber });
  };

  /**
   * useEffect
   * _______________________________________________________________________________
   */
  useEffect(() => {
    reset();
    if (step === "recivecode") {
      pause();
    }
    if (step === "verificationcode") {
      start();
      handleWebOTP();
    }
  }, [step]);

  /**
   * template
   * _______________________________________________________________________________
   */

  return (
    <>
      {step === "recivecode" ? (
        <Root errorTesxt={errors?.mobileNumber?.message}>
          <StickyHeader>
            <Image
              loader={imageLoader as any}
              src={THISPROJECT.LOGO_URL}
              alt={"header-logo"}
              width={100}
              height={30}
              className="logo"
              priority
            />
          </StickyHeader>
          <Typography textalign="center" fontWeight={700}>
            شماره موبایل خود را وارد کنید
          </Typography>
          <InputContainer errorTesxt={errors.mobileNumber?.message} className="InputContainer">
            <input
              style={{
                outline: "none",
                backgroundColor: "#fff",
                height: "54px",
                borderRadius: "8px",
                padding: "4px",
                marginTop: "9px",
                width: "100%",
                fontSize: "16px",
                WebkitAppearance: "none",
                WebkitTapHighlightColor: "transparent",
                border: "1px solid #000",
                color: "inherit",
                fontFamily: "inherit",
              }}
              type="text"
              inputMode="numeric"
              autoComplete={"off"}
              autoCorrect="off"
              autoCapitalize="none"
              spellCheck={false}
              placeholder="مثال : ۰۹۱۲۰۰۰۰۰۰۰"
              {...register("mobileNumber", {
                onChange(event) {
                  setValue("mobileNumber", event.target.value.slice(0, 11));
                },
              })}
            />
            {errors.mobileNumber?.message && <IoMdInformationCircleOutline />}
          </InputContainer>
          <TextError>{errors.mobileNumber?.message}</TextError>
          <Button
            onClick={() => {
              sendCodeMutate({ mobileNumber: watch("mobileNumber") as string });
            }}
            variant="checkout"
            position="fixed"
            disabled={!!errors.mobileNumber?.message || !watch("mobileNumber")}
          >
            {loadingMutate ? (
              <AppFlex justify="center">
                <ReactLoading color="#ffff" type="spin" width={20} height={20} />
              </AppFlex>
            ) : (
              <Typography>تایید وادامه</Typography>
            )}
          </Button>
        </Root>
      ) : (
        step === "verificationcode" && (
          <Root errorTesxt={errors?.VerificationCode?.message}>
            <StickyHeader>
              <Image
                loader={imageLoader as any}
                src={THISPROJECT.LOGO_URL}
                alt={"header-logo"}
                width={100}
                height={30}
                className="logo"
                priority
              />
            </StickyHeader>
            <AppFlex direction="column" align="center" gap={"10px"}>
              <Typography variant="body1" textalign="center" fontWeight={700}>
                کد تایید را وارد کنید
              </Typography>
              <Typography variant="subtitle2">{`کد 5 رقمی به شماره ${userData.phNumber} ارسال شده است.`}</Typography>
            </AppFlex>
            <InputContainer errorTesxt={errors.VerificationCode?.message} className="InputContainer">
              <input
                style={{
                  outline: "none",
                  backgroundColor: "#fff",
                  height: "54px",
                  borderRadius: "8px",
                  padding: "4px",
                  marginTop: "9px",
                  width: "100%",
                  fontSize: "16px",
                  WebkitAppearance: "none",
                  WebkitTapHighlightColor: "transparent",
                  border: "1px solid #000",
                  color: "inherit",
                  fontFamily: "inherit",
                }}
                type="text"
                inputMode="numeric"
                placeholder="مثال : ۹۲۴۵۴"
                autoComplete={"off"}
                autoCorrect="off"
                autoCapitalize="none"
                spellCheck={false}
                maxLength={5}
                {...register("VerificationCode", {
                  onChange(event) {
                    setValue("VerificationCode", event.target.value.slice(0, 5));
                  },
                })}
              />
              {errors.VerificationCode?.message && <IoMdInformationCircleOutline />}
            </InputContainer>
            <TextError>{errors.VerificationCode?.message}</TextError>

            {/* EditAndResetTimer */}
            <EditAndResetTimer direction="column" gap={"10px"}>
              <AppFlex>
                <IoMdLock fill="#AAB3C0" />
                <ResetTmeBtn disabled={isStarted} className="typo" onClick={handleResetCode}>
                  درخواست مجدد ارسال کد
                </ResetTmeBtn>
                <Typography textcolor="#AAB3C0" className="typo">
                  {render(time)}
                </Typography>
              </AppFlex>
              <AppFlex align="center">
                <MdOutlineModeEdit fill="red" />
                <Typography
                  className="typo edit-btn"
                  variant="subtitle2"
                  textcolor="red"
                  onClick={() => setStep("recivecode")}
                >
                  ویرایش شماره موبایل
                </Typography>
              </AppFlex>
            </EditAndResetTimer>
            <Button
              onClick={() => {
                mutate({
                  verificationCode: watch("VerificationCode") as string,
                  mobileNumber: userData.phNumber,
                  code: code,
                });
              }}
              variant="question"
              position="fixed"
              disabled={!!errors.VerificationCode?.message || !watch("VerificationCode")}
            >
              {isLoading ? (
                <AppFlex justify="center">
                  <ReactLoading color="#ffff" type="spin" width={20} height={20} />
                </AppFlex>
              ) : (
                <Typography variant="subtitle1">مشاهده برنامه</Typography>
              )}
            </Button>
          </Root>
        )
      )}
    </>
  );
};

export default ConfirmPAyPage;

const Root = styled.section<{ errorTesxt: string | undefined }>`
  padding-inline: 16px;
  position: relative;
  padding-block: 100px 40px;
  max-width: 600px;
  margin: auto;
  display: flex;
  flex-direction: column;
  gap: 20px;
  direction: rtl;

  input {
    ::placeholder {
      opacity: 50%;
    }
  }

`;

const InputContainer = styled.div<{ errorTesxt: string | undefined }>`
  p {
    color: ${({ errorTesxt }) => (errorTesxt ? `red` : "#000")};
    font-size: 14px;
  }

  svg {
    scale: 1.4;
    fill: red;
  }
`;

const TextError = styled.p`
  position: absolute;
  bottom: 35px;
  font-size: 10px;
  color: red;
  right: 25px;
`;

const EditAndResetTimer = styled(AppFlex)`
  padding-block: 20px;
  .typo {
    font-size: 12px;
    margin-right: 5px;
  }

  .edit-btn {
    cursor: pointer;
  }
`;

const ResetTmeBtn = styled.button`
  border-style: none;
  appearance: none;
  background-color: #fff;
  cursor: pointer;

  &:disabled {
    box-shadow: none;
    cursor: not-allowed;
  }
`;

const StickyHeader = styled.header`
  position: fixed;
  width: 100%;
  padding: 16px;
  top: 0;
  left: 50%;
  translate: -50%;
  display: flex;
  justify-content: center;
  align-items: center;
  background-color: #fff;
  z-index: 95;

  .logo {
    margin-inline: auto;
  }
`;
